package com.liquidnet.service.platform.service.ticketSystem;

import com.liquidnet.service.platform.vo.ticketSystem.*;

import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>
 * 票务对接平台 服务类
 * </p>
 *
 * @author liquidnet
 * @since 2021-07-20
 */
public interface ITicketSystemService {

    /**
     * 获取accessToken
     *
     * @return
     */
    STAccessTokenVo getAccessToken();

    /**
     * 查询标准演出类型
     *
     * @param accessToken 访问令牌
     * @return
     */
    List<STPTListVo> getPerformanceTypeList(String accessToken);

    /**
     * 查询标准演出场所
     *
     * @param accessToken 访问令牌
     * @param provinceId  省ID
     * @param cityId      市ID
     * @param districtId  区ID
     * @param pageNo      页码，默认1
     * @param pageSize    每页条数最大200
     * @param venueName   场所名称，支持模糊查询
     * @return
     */
    List<STFieldListVo> getFieldList(String accessToken, String provinceId, String cityId, String districtId, Integer pageNo, Integer pageSize, String venueName);

    /**
     * 新增演出场厅
     *
     * @param accessToken       访问令牌
     * @param venueCode         场所编码
     * @param provinceId        省ID
     * @param cityId            市ID
     * @param districtId        区ID
     * @param venueName         场所名称
     * @param venueAddress      场所详细地址
     * @param managementCompany 经营单位
     * @param hasHall           是否有场厅
     * @param hallName          场厅名称
     * @param hasSeat           是否有座
     * @param seatNum           场厅座位数量
     * @param seatingCapacity   场厅可容纳人数
     */
    STFieldChangeVo insertField(String accessToken, String venueCode, Integer provinceId, Integer cityId,
                                Integer districtId, String venueName, String venueAddress, String managementCompany,
                                Integer hasHall, String hallName, Integer hasSeat, Integer seatNum,
                                Integer seatingCapacity);


    /**
     * 上报演出场次接口
     *
     * @param accessToken     访问令牌
     * @param performanceId   演出活动ID
     * @param performanceName 演出活动名称
     * @param approval        批文号
     * @param typeId          演出类型编码，参考附录演出类型
     * @param posterUrl       演出封面图
     * @param performanceDesc 演出描述
     * @param sponsor         主办方，多个用英文逗号分隔
     * @param performer       演员，多个用英文逗号分隔
     * @param sessionList     演出场次ID List，传多个场次时可以批量获取场次编码
     * @return {
     * "sessionId1":"sessionCode1",
     * "sessionId2":"sessionCode2"
     * }
     */
    String insertPerformance(String accessToken, String performanceId, String performanceName, String approval, Integer typeId,
                             String posterUrl, String performanceDesc, String sponsor, String performer, List<TimesVo> sessionList);


    /**
     * 更新上报的演出项目数据
     *
     * @param accessToken     访问令牌
     * @param performanceId   演出ID
     * @param performanceName 演出名称
     * @param performanceDesc 演出描述
     * @param typeId          演出类型
     * @param posterUrl       演出封面图
     * @param sponsor         主办方，多个用英文逗号分隔
     * @param performer       演员，多个用英文逗号分隔
     * @param approval        批文号
     */
    boolean updatePerformance(String accessToken, String performanceId, String performanceName, String performanceDesc, Integer typeId,
                              String posterUrl, String sponsor, String performer, String approval);

    /**
     * 更新上报的场次数据
     *
     * @param accessToken      访问令牌
     * @param sessionId        演出场次ID
     * @param sessionCode      演出场次编码
     * @param sessionName      场次名称
     * @param sessionStartTime 场次开始时间
     * @param sessionEndTime   场次结束时间
     * @param hallCode         场厅编码
     * @param hasSeat          是否有座，0：无座1：有座
     * @param performanceId    演出ID
     * @param sessionStatus    场次状态，1 正常，2 取消
     * @param tag              标记，0:正式，1:测试
     * @param seatingChart     票图URL
     * @return
     */
    boolean updateTimes(String accessToken, String sessionId, String sessionCode, String sessionName, LocalDateTime sessionStartTime,
                        LocalDateTime sessionEndTime, String hallCode, Integer hasSeat, String performanceId, Integer sessionStatus,
                        Integer tag, String seatingChart);

    /**
     * 上报票价信息
     *
     * @param accessToken 访问令牌
     * @param sessionCode 场次编码
     * @param priceList   票价信息数组
     * @return
     */
    boolean insertTicketPrice(String accessToken, String sessionCode, List<TicketPriceVo> priceList);

    /**
     * 删除票价信息
     *
     * @param accessToken   访问令牌
     * @param performanceId 演出ID
     * @param sessionCode   场次编码
     * @param priceId       票价ID
     * @return
     */
    boolean delTicketPrice(String accessToken, String performanceId, String sessionCode, String priceId);

    /**
     * 上报票信息
     *
     * @param accessToken   访问令牌
     * @param performanceId 演出ID
     * @param sessionId     场次id
     * @param sessionCode   场次编码
     * @param priceId       票价ID
     * @param ticketList    票List
     * @return
     */
    boolean insertTicket(String accessToken, String performanceId, String sessionId, String sessionCode, String priceId, List<TicketVo> ticketList);

    /**
     * 删除票信息
     *
     * @param accessToken   访问令牌
     * @param performanceId 演出id
     * @param sessionCode   场次编码
     * @param priceId       票价ID
     * @param ticketId      票ID
     * @return
     */
    boolean delTicket(String accessToken, String performanceId, String sessionCode, String priceId, String ticketId);

    /**
     * 上报订单信息
     *
     * @param accessToken     访问令牌
     * @param orderId         订单id
     * @param orderTime       下单时间
     * @param channelType     渠道类型，1自销，2分销
     * @param channelName     渠道名称
     * @param ticketNum       购票数量
     * @param totalAmount     订单总金额，单位分，订单实收金额
     * @param ticketType      出票类型1：正常票，2：工作票，3：赠票
     * @param ticketMode      出票方式, 1纸制票，2 电子票
     * @param authMode        实名方式,1 非实名，2一单一证，3一票一证
     * @param ticketOrderList 票单List，多个票用英文逗号分隔
     * @return
     */
    boolean insertOrder(String accessToken, String orderId, LocalDateTime orderTime, Integer channelType, String channelName, Integer ticketNum, long totalAmount, Integer ticketType, Integer ticketMode, Integer authMode, List<OrderVo> ticketOrderList);

    /**
     * 上报退票信息
     *
     * @param accessToken     访问令牌
     * @param refundOrderId   退单订单id
     * @param orderId         订单id
     * @param reason          退单原因
     * @param refundOrderTime 退单时间
     * @param ticketOrderList 票单List
     * @return
     */
    boolean refundOrder(String accessToken, String refundOrderId, String orderId, String reason, LocalDateTime refundOrderTime, List<RefundOrderVo> ticketOrderList);

    /**
     * 上报检票数据
     *
     * @param accessToken 访问令牌
     * @param ticketList  票信息List
     * @return
     */
    boolean insertCheck(String accessToken, List<CheckVo> ticketList);
}
