package com.liquidnet.service.goblin.dto.manage;

import com.liquidnet.commons.lang.constant.LnsRegex;
import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.DateUtil;
import com.liquidnet.commons.lang.util.IDGenerator;
import com.liquidnet.service.goblin.dto.GoblinGoodsSpecDto;
import com.liquidnet.service.goblin.dto.vo.GoblinGoodsInfoVo;
import com.liquidnet.service.goblin.dto.vo.GoblinGoodsSkuInfoVo;
import com.liquidnet.service.goblin.dto.vo.GoblinGoodsSpecValueVo;
import com.liquidnet.service.goblin.dto.vo.GoblinGoodsSpecVo;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import javax.validation.constraints.Pattern;
import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

@ApiModel(value = "GoblinStoreMgtGoodsAddParam", description = "商品管理:添加商品入参")
@Data
public class GoblinStoreMgtGoodsAddParam implements Serializable {
    private static final long serialVersionUID = -5101611616626164702L;
    @ApiModelProperty(position = 10, required = false, value = "商品ID[编辑时必传]")
    private String spuId;
    @ApiModelProperty(position = 11, required = true, value = "店铺ID")
    private String storeId;

    /**
     * ---------------------------- 基本信息 ----------------------------
     **/

    @ApiModelProperty(position = 12, required = true, value = "商品名称[128]")
    private String name;
    @ApiModelProperty(position = 13, required = false, value = "商品名称[128]")
    private String subtitle;
    @ApiModelProperty(position = 14, required = false, value = "商品销售价[20,2]")
    private BigDecimal sellPrice;

    @ApiModelProperty(position = 15, required = true, value = "商品一级分类ID[30]")
    private String cateFid;
    @ApiModelProperty(position = 16, required = true, value = "商品二级分类ID[30]")
    private String cateSid;
    @ApiModelProperty(position = 17, required = true, value = "商品三级分类ID[30]")
    private String cateTid;

    @ApiModelProperty(position = 18, required = false, value = "商品简介[256]", example = "商品简介...")
    private String intro;

    @ApiModelProperty(position = 19, required = false, value = "商品标签列表")
    private List<String> tagList;
    @ApiModelProperty(position = 20, required = true, value = "商品图片列表")
    private List<String> imageList;
    @ApiModelProperty(position = 21, required = false, value = "商品视频地址")
    private String coverPic;
    @ApiModelProperty(position = 22, required = false, value = "商品视频地址")
    private String video;

    /**
     * ---------------------------- 价格库存 ----------------------------
     **/

    @ApiModelProperty(position = 23, required = true, value = "规格展现方式[1-常规|2-组合]", allowableValues = "1,2", example = "1")
    private String specMode;
    @ApiModelProperty(position = 24, required = true, value = "SKU规格信息")
    private List<GoblinStoreMgtGoodsAddSkuParam> skuParamList;
    @ApiModelProperty(position = 25, required = true, value = "物流模版", example = "松鼠德邦")
    private String logisticsTemplate;
    @ApiModelProperty(position = 26, required = false, value = "商品编码[默认为系统编码，也可手动输入商家自己的编码]")
    private String spuNo;

    /**
     * ---------------------------- 商品介绍 ----------------------------
     **/

    @ApiModelProperty(position = 27, required = true, value = "商品详情", example = "商品详情...")
    private String details;

    /**
     * ---------------------------- 销售属性 ----------------------------
     **/

    @ApiModelProperty(position = 28, required = true, value = "上架处理方式[1-等待手动上架|2-直接上架售卖|3-预约定时上架]", example = "1")
    private String shelvesHandle;
    @ApiModelProperty(position = 29, required = false, value = "预约上架时间[上架处理方式为3-预约定时上架时需要指定]")
    @Pattern(regexp = LnsRegex.Valid.DATETIME_FULL, message = "预约上架时间格式有误")
    private String shelvesTime;
    @ApiModelProperty(position = 30, required = false, value = "商品有效期[YYYY-MM-DD]")
    @Pattern(regexp = LnsRegex.Valid.DATETIME_YMD, message = "商品有效期格式有误")
    private String spuValidity;
    @ApiModelProperty(position = 31, required = true, value = "是否虚拟商品[0-否|1-是]", allowableValues = "0,1", example = "0")
    @Pattern(regexp = "\\b(0|1)\\b", message = "是否虚拟商品参数无效")
    private String virtualFlg;

    /**
     * ---------------------------- 服务保障 ----------------------------
     **/

    @ApiModelProperty(position = 32, required = false, value = "商品服务支持ID列表")
    private List<String> ssidList;

    /**
     * ---------------------------- 专属标签 ----------------------------
     **/

    @ApiModelProperty(position = 33, required = true, value = "商品关联音乐人、艺术家、IP列表")
    private List<String> extagList;


    /**
     * 生成SPU
     *
     * @return GoblinGoodsInfoVo
     */
    public GoblinGoodsInfoVo initGoodsInfoVo() {
        GoblinGoodsInfoVo vo = GoblinGoodsInfoVo.getNew();
        vo.setStoreId(this.getStoreId());
        if (StringUtils.isBlank(this.getSpuId())) {
            vo.setSpuId(IDGenerator.nextMilliId2());
        } else {
            vo.setSpuId(this.getSpuId());
        }
        if (StringUtils.isBlank(this.getSpuNo())) {
            vo.setSpuNo(IDGenerator.nextTimeId2());
        } else {
            vo.setSpuNo(this.getSpuNo());
        }
        vo.setName(this.getName());
        vo.setSubtitle(this.getSubtitle());
        vo.setSellPrice(this.getSellPrice());
//        vo.setPriceGe(BigDecimal.ZERO);
//        vo.setPriceLe(BigDecimal.ZERO);
        vo.setIntro(this.getIntro());
        vo.setDetails(this.getDetails());
        vo.setCoverPic(this.getCoverPic());
        vo.setVideo(this.getVideo());
        vo.setSpecMode(this.getSpecMode());
        vo.setCateFid(this.getCateFid());
        vo.setCateSid(this.getCateSid());
        vo.setCateTid(this.getCateTid());
//        vo.setStoreCateFid(null);
//        vo.setStoreCateSid(null);
//        vo.setStoreCateTid(null);
//        vo.setBrandId(null);
        vo.setShelvesHandle(this.getShelvesHandle());
        vo.setShelvesTime(this.getShelvesTime());
        vo.setSpuValidity(this.getSpuValidity());
        vo.setVirtualFlg(this.getVirtualFlg());
        vo.setStatus("1");
//        vo.setReason(null);
        vo.setShelvesStatus("0");
//        vo.setSpuAppear(null);
//        vo.setShelvesAt(null);
        vo.setImageList(this.getImageList());
        vo.setLogisticsTemplate(this.getLogisticsTemplate());
//        vo.setCreatedBy();
//        vo.setCreatedAt();
//        vo.setUpdatedBy();
//        vo.setUpdatedAt();
//        vo.setSpecVoList();
//        vo.setTagVoList();
//        vo.setServiceSupportVoList();
//        vo.setExtagVoList();
        return vo;
    }

    /**
     * 编辑SPU参数整理
     *
     * @return GoblinGoodsInfoVo
     */
    public GoblinGoodsInfoVo initEditGoodsSkuInfoVo() {
        GoblinGoodsInfoVo vo = GoblinGoodsInfoVo.getNew();
        vo.setStoreId(this.getStoreId());
        vo.setSpuId(this.getSpuId());
        vo.setSpuNo(this.getSpuNo());
        vo.setName(this.getName());
        vo.setSubtitle(this.getSubtitle());
        vo.setSellPrice(this.getSellPrice());
        vo.setIntro(this.getIntro());
        vo.setDetails(this.getDetails());
        vo.setCoverPic(this.getCoverPic());
        vo.setVideo(this.getVideo());
        vo.setSpecMode(this.getSpecMode());
        vo.setCateFid(this.getCateFid());
        vo.setCateSid(this.getCateSid());
        vo.setCateTid(this.getCateTid());
        vo.setShelvesHandle(this.getShelvesHandle());
        vo.setShelvesTime(this.getShelvesTime());
        vo.setSpuValidity(this.getSpuValidity());
        vo.setVirtualFlg(this.getVirtualFlg());
        vo.setImageList(this.getImageList());
        vo.setLogisticsTemplate(this.getLogisticsTemplate());
        return vo;
    }

    /**
     * 生成SKU列表以及SPU规格等信息
     *
     * @param goodsInfoVo  GoblinGoodsInfoVo
     * @param vos          List<GoblinGoodsSkuInfoVo>
     * @param goodsSpecVos List<GoblinGoodsSpecVo>
     */
    public void initGoodsSkuInfoVo(GoblinGoodsInfoVo goodsInfoVo, List<GoblinGoodsSkuInfoVo> vos,
                                   List<GoblinGoodsSpecVo> goodsSpecVos) {
        List<GoblinStoreMgtGoodsAddSkuParam> mgtGoodsAddSkuParamList = this.getSkuParamList();
        int size = mgtGoodsAddSkuParamList.size();

        ArrayList<String> skuIdList = CollectionUtil.arrayListString();
        BigDecimal priceGe = BigDecimal.ZERO, priceLe = BigDecimal.ZERO;
        for (int i = 0; i < size; i++) {
            GoblinStoreMgtGoodsAddSkuParam addSkuParam = mgtGoodsAddSkuParamList.get(i);
            String skuId = addSkuParam.getSkuId();

            GoblinGoodsSkuInfoVo vo = GoblinGoodsSkuInfoVo.getNew();
            if (StringUtils.isBlank(skuId)) {
                vo.setSkuId(goodsInfoVo.getSpuId().concat(StringUtils.right(String.valueOf(System.nanoTime()), 5)));
            } else {
                vo.setSkuId(skuId);
            }
            vo.setSpuId(goodsInfoVo.getSpuId());
//            vo.setSkuNo(goodsInfoVo.getSpuNo().concat("-").concat(String.valueOf(i)));
            vo.setName(goodsInfoVo.getName());
            vo.setSubtitle(goodsInfoVo.getSubtitle());
            vo.setSkuPic(addSkuParam.getSkuPic());
            vo.setSkuIsbn(addSkuParam.getSkuIsbn());
            vo.setStock(addSkuParam.getStock());
            vo.setSkuStock(vo.getStock());
            vo.setWarningStock(addSkuParam.getWarningStock());
            vo.setSellPrice(addSkuParam.getSellPrice());
            vo.setPrice(addSkuParam.getPrice());
            vo.setPriceMember(addSkuParam.getPriceMember());
            vo.setWeight(addSkuParam.getWeight());
            vo.setBuyFactor(addSkuParam.getBuyFactor());
            vo.setBuyRoster(addSkuParam.getBuyRoster());
            vo.setBuyLimit(addSkuParam.getBuyLimit());
            vo.setStoreId(goodsInfoVo.getStoreId());
            vo.setSkuValidity(DateUtil.Formatter.yyyyMMddHHmmss.parse(addSkuParam.getSkuValidity()));
            vo.setVirtualFlg(goodsInfoVo.getVirtualFlg());
            vo.setStatus("3");// 没有审核流程，默认通过
//            vo.setReason();
            vo.setShelvesStatus("0");
            vo.setSkuAppear(addSkuParam.getSkuAppear());
            vo.setSkuNo("0");
//            vo.setShelvesAt();
            vo.setCreatedBy(goodsInfoVo.getCreatedBy());
            vo.setCreatedAt(goodsInfoVo.getCreatedAt());
//            vo.setUpdatedBy();
//            vo.setUpdatedAt();
            vo.setLogisticsTemplate(goodsInfoVo.getLogisticsTemplate());

            List<GoblinGoodsSpecDto> skuSpecList = addSkuParam.getSkuSpecList();
            vo.setSkuSpecList(skuSpecList);

            vos.add(vo);
            skuIdList.add(vo.getSkuId());

            priceGe = priceGe.compareTo(vo.getPrice()) > 0 ? vo.getPrice() : priceGe;
            priceLe = priceLe.compareTo(vo.getPrice()) < 0 ? vo.getPrice() : priceLe;

            int specSize = skuSpecList.size();// SKU包含的规格数量
            for (int j = 0; j < specSize; j++) {
                GoblinGoodsSpecDto specDto = skuSpecList.get(j);

                GoblinGoodsSpecValueVo specValueVo = GoblinGoodsSpecValueVo.getNew().setSpecVname(specDto.getSpecVname()).setSpecVsort(0);

                if (CollectionUtils.isEmpty(goodsSpecVos)) {// 初始化生成商品规格
                    ArrayList<GoblinGoodsSpecValueVo> specValueVos = new ArrayList<>();
                    specValueVos.add(specValueVo);

                    goodsSpecVos.add(GoblinGoodsSpecVo.getNew()
                            .setSpecName(specDto.getSpecName()).setSpecSort(0).setSpecValues(specValueVos));
                } else {// 补充规格对应的值
                    goodsSpecVos.get(i).getSpecValues().add(specValueVo);
                }
            }
        }
        goodsInfoVo.setPriceGe(priceGe);
        goodsInfoVo.setPriceLe(priceLe);
        goodsInfoVo.setSpecVoList(goodsSpecVos);
        goodsInfoVo.setSkuIdList(skuIdList);
    }
}
