package com.liquidnet.service.goblin.service.impl;

import com.liquidnet.commons.lang.util.*;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.base.SqlMapping;
import com.liquidnet.service.base.constant.MQConst;
import com.liquidnet.service.goblin.dto.vo.*;
import com.liquidnet.service.goblin.enums.OrderStatus;
import com.liquidnet.service.goblin.enums.PayStatus;
import com.liquidnet.service.goblin.param.GoblinBraceletOrderPayParam;
import com.liquidnet.service.goblin.param.dougong.DougongJsPayData;
import com.liquidnet.service.goblin.param.dougong.DougongRequestParam;
import com.liquidnet.service.goblin.service.IGoblinBraceLetOrderService;
import com.liquidnet.service.goblin.service.IGoblinDougongPayService;
import com.liquidnet.service.goblin.service.IGoblinRechargeWristbandService;
import com.liquidnet.service.goblin.util.GoblinRedisUtils;
import com.liquidnet.service.goblin.util.QueueUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class GoblinBraceLetOrderServiceImpl implements IGoblinBraceLetOrderService {

    @Autowired
    GoblinRedisUtils goblinRedisUtils;
    @Autowired
    private QueueUtils queueUtils;
    @Autowired
    private IGoblinRechargeWristbandService rechargeWristbandService;
    @Autowired
    private IGoblinDougongPayService dougongPayService;


    @Override
    public ResponseDto<GoblinBraceletPayResultVo> checkOrder(GoblinBraceletOrderPayParam payParam) {
        log.info("request param: {}.", JsonUtils.toJson(payParam));
        // 1. 检查商品
        GoblinRechargeWristbandVo rechargeWristbandVo = rechargeWristbandService.getList();
        if (rechargeWristbandVo == null) {
            log.error("查询商品为空");
            return ResponseDto.failure("下单失败");
        }
        // 2. 计算金额
        if (!rechargeWristbandVo.getWristbandId().equals(payParam.getWristbandId())) {
            log.error("下单商品不存在");
            return ResponseDto.failure("下单商品不存在");
        }

        List<GoblinRechargeAmountVo> amonutList = rechargeWristbandVo.getAmonutList();
        if (amonutList.isEmpty()) {
            log.error("充值面额不存在");
            return ResponseDto.failure("充值面额不存在");
        }

        List<GoblinRechargeAmountVo> payAmountList = amonutList.stream()
                .filter(a -> a.getAmountId().equals(payParam.getAmountId()))
                .collect(Collectors.toList());

        if (payAmountList.isEmpty()) {
            log.error("充值面额不存在");
            return ResponseDto.failure("充值面额不存在");
        }

        // 要充值的面额
        final GoblinRechargeAmountVo payAmountVo = payAmountList.get(0);

        final GoblinBraceletOrderVo orderVo = getOrderVo(rechargeWristbandVo.getWristbandId(),
                rechargeWristbandVo.getName(),
                rechargeWristbandVo.getPrice(),
                payAmountVo, payParam);

        if (orderVo == null) {
            log.error("下单失败");
            return ResponseDto.failure("下单失败");
        }

        // 4. 调用斗拱拉起支付
        DougongJsPayData dougongJsPayData = new DougongJsPayData();
        dougongJsPayData.setReqDate(orderVo.getReqDate());
        dougongJsPayData.setReqSeqId(orderVo.getReqSeqId());
        dougongJsPayData.setTradeType(orderVo.getTradeType());
        dougongJsPayData.setGoodsDesc(orderVo.getGoodsDesc());
        dougongJsPayData.setTransAmt(orderVo.getPriceTotal().setScale(2, RoundingMode.HALF_UP).toPlainString());
        dougongJsPayData.setTimeExpire(DateUtil.format(DateUtil.addHour(new Date(), 2), DateUtil.Formatter.yyyyMMddHHmmssTrim));
        dougongJsPayData.setSubOpenid(payParam.getSubOpenId());
        dougongJsPayData.setGoodsDetailList(getGoodsDetailList(orderVo.getOrderEntitiesDetailVoList()));

        Map<String, Object> dougongResponseMap = dougongPayService.jsPay(dougongJsPayData);
        if (null == dougongResponseMap || dougongResponseMap.isEmpty()) {
            log.error("拉起下单失败");
            return ResponseDto.failure("下单失败");
        }

        // 根据响应接口填充参数
        orderVo.setHfSeqId((String) dougongResponseMap.get("hf_seq_id"));
        orderVo.setAtuSubMerId((String) dougongResponseMap.get("atu_sub_mer_id"));
//        orderVo.setOutTransId(dougongResponseMap.get(""));
        orderVo.setPartyOrderId((String) dougongResponseMap.get("party_order_id"));
//        orderVo.setEndTime(dougongResponseMap.get(""));
//        orderVo.setAcctDate(dougongResponseMap.get(""));
//        orderVo.setPriceActual(dougongResponseMap.get(""));

        // 5. 根据响应结构写 redis、mysql
        goblinRedisUtils.setBraceletOrderVo(orderVo);

        LinkedList<String> sqls = CollectionUtil.linkedListString();
        sqls.add(SqlMapping.get("goblin_bracelet_order_insert"));
        sqls.add(SqlMapping.get("goblin_bracelet_order_entities_insert"));
        LinkedList<Object[]> sqlDataOrder = CollectionUtil.linkedListObjectArr();


        sqlDataOrder.add(new Object[]{
                orderVo.getOrderId(), orderVo.getUserId(), orderVo.getBindName(), orderVo.getBindMobile(), orderVo.getBindIdcard(), orderVo.getReqDate(), orderVo.getGoodsDesc(),
                orderVo.getReqSeqId(), orderVo.getHfSeqId(), orderVo.getTradeType(), orderVo.getAtuSubMerId(), orderVo.getPartyOrderId(),
                orderVo.getPrice(), orderVo.getPriceTotal(), orderVo.getPriceRefund(), orderVo.getRefundPriceCharges(),
                orderVo.getRefundNumber(), orderVo.getStatus(), orderVo.getPayStatus(), orderVo.getCreatedAt(), orderVo.getUpdatedAt()
        });
        LinkedList<Object[]> sqlDataOrderEntities = CollectionUtil.linkedListObjectArr();
        for (GoblinBraceletOrderEntitiesVo entitiesVo : orderVo.getOrderEntitiesDetailVoList()) {
            sqlDataOrderEntities.add(new Object[]{
                    entitiesVo.getOrderEntitiesId(), entitiesVo.getOrderId(), entitiesVo.getGoodsType(), entitiesVo.getGoodsId(), entitiesVo.getGoodsName(),
                    entitiesVo.getGoodsPrice(), entitiesVo.getStatus(), entitiesVo.getPayStatus(), entitiesVo.getCreatedAt(),
                    entitiesVo.getUpdatedAt()
            });
        }

        queueUtils.sendMsgByRedis(
                MQConst.GoblinQueue.SQL_GOODS.getKey(),
                SqlMapping.gets(sqls, sqlDataOrder, sqlDataOrderEntities)
        );

        // 6. 响应数据
        GoblinBraceletPayResultVo resultVo = GoblinBraceletPayResultVo.getNew();
        resultVo.setOrderId(orderVo.getOrderId());
        resultVo.setPrice(orderVo.getPriceTotal());
        resultVo.setPayInfo(dougongResponseMap.get("pay_info"));
        return ResponseDto.success(resultVo);
    }

    private List<DougongRequestParam.WxDataGoodsDetail> getGoodsDetailList(List<GoblinBraceletOrderEntitiesVo> orderEntitiesDetailVoList) {
        if (orderEntitiesDetailVoList.isEmpty()) {
            return Collections.emptyList();
        }
        List<DougongRequestParam.WxDataGoodsDetail> goodsDetailList = new ArrayList<>(orderEntitiesDetailVoList.size());
        for (GoblinBraceletOrderEntitiesVo entitiesVo : orderEntitiesDetailVoList) {
            DougongRequestParam.WxDataGoodsDetail detail = new DougongRequestParam.WxDataGoodsDetail();
            detail.setGoodsId(entitiesVo.getGoodsId());
            detail.setGoodsName(entitiesVo.getGoodsName());
            detail.setPrice(entitiesVo.getGoodsPrice().toPlainString());
            detail.setQuantity(1);
            goodsDetailList.add(detail);
        }
        return goodsDetailList;
    }

    /**
     * @param wristbandId 手环ID
     * @param name        手环名称
     * @param price       手环价格
     * @param payAmountVo 充值面额VO
     * @param payParam    请求参数
     * @return orderId
     */
    private GoblinBraceletOrderVo getOrderVo(String wristbandId,
                                             String name,
                                             BigDecimal price,
                                             GoblinRechargeAmountVo payAmountVo,
                                             GoblinBraceletOrderPayParam payParam) {
        // 需付款金额
        BigDecimal totalAmount = BigDecimal.ZERO.add(price).add(payAmountVo.getPrice());

        final String uid = CurrentUtil.getCurrentUid();
        log.info("用户下单, 需要支付金额: {}, uid: {}.", totalAmount.toPlainString(), uid);

        final String orderId = IDGenerator.nextSnowId();

        try {
            List<GoblinBraceletOrderEntitiesVo> list = new ArrayList<>();
            list.add(buildRechargeOrderEntitiesVo(wristbandId, name, price, 1, orderId));
            list.add(buildRechargeOrderEntitiesVo(payAmountVo.getAmountId(), payAmountVo.getName(), payAmountVo.getPrice(), 2, orderId));

            GoblinBraceletOrderVo orderVo = buildRechargeOrder(payParam, orderId, uid, totalAmount);
            orderVo.setOrderEntitiesDetailVoList(list);
            return orderVo;
        } catch (Exception e) {
            log.error("error", e);
            return null;
        }
    }

    private GoblinBraceletOrderVo buildRechargeOrder(GoblinBraceletOrderPayParam payParam, String orderId, String uid, BigDecimal totalAmount) {
        GoblinBraceletOrderVo braceletOrderVo = new GoblinBraceletOrderVo();
        braceletOrderVo.setOrderId(orderId);
        braceletOrderVo.setUserId(uid);
        braceletOrderVo.setBindName(payParam.getName());
        braceletOrderVo.setBindMobile(payParam.getMobile());
        braceletOrderVo.setBindIdcard(payParam.getIdCard());
        braceletOrderVo.setReqDate(DateUtil.getNowTime("yyyyMMdd"));
        braceletOrderVo.setGoodsDesc(getGoodsDesc());
        braceletOrderVo.setReqSeqId(IDGenerator.nextSnowId());
        braceletOrderVo.setTradeType("T_MINIAPP");
        braceletOrderVo.setPrice(totalAmount);
        braceletOrderVo.setPriceTotal(totalAmount);
        braceletOrderVo.setPriceRefund(BigDecimal.ZERO);
        braceletOrderVo.setRefundPriceCharges(BigDecimal.ZERO);
        braceletOrderVo.setRefundNumber(0);
        braceletOrderVo.setStatus(OrderStatus.PENDING_PAYMENT.getCode());
        braceletOrderVo.setPayStatus(PayStatus.NOT_PAID.getCode());
        braceletOrderVo.setCreatedAt(LocalDateTime.now());
        braceletOrderVo.setUpdatedAt(LocalDateTime.now());
        return braceletOrderVo;
    }

    private GoblinBraceletOrderEntitiesVo buildRechargeOrderEntitiesVo(String goodsId, String goodsName, BigDecimal goodsPrice, Integer goodsType, String orderId) {
        GoblinBraceletOrderEntitiesVo entitiesVo = new GoblinBraceletOrderEntitiesVo();
        entitiesVo.setOrderId(orderId);
        entitiesVo.setOrderEntitiesId(IDGenerator.nextSnowId());
        entitiesVo.setGoodsType(goodsType);
        entitiesVo.setGoodsId(goodsId);
        entitiesVo.setGoodsName(goodsName);
        entitiesVo.setGoodsPrice(goodsPrice);
        entitiesVo.setStatus(OrderStatus.PENDING_PAYMENT.getCode());
        entitiesVo.setPayStatus(PayStatus.NOT_PAID.getCode());
        entitiesVo.setCreatedAt(LocalDateTime.now());
        entitiesVo.setUpdatedAt(LocalDateTime.now());
        return entitiesVo;
    }

    /**
     * 获取商品描述
     *
     * @return
     */
    private String getGoodsDesc() {
        return "手环充值";
    }

    @Override
    public ResponseDto<Integer> checkOrderResult(String orderId) {
        final String uid = CurrentUtil.getCurrentUid();
        GoblinBraceletOrderVo braceletOrderVo = goblinRedisUtils.getBraceletOrderVo(uid, orderId);
        if (braceletOrderVo == null) {
            log.error("订单不存在");
            return ResponseDto.failure("订单不存在");
        }
        return ResponseDto.success(braceletOrderVo.getPayStatus());
    }

    @Override
    public ResponseDto<Boolean> dougongCallBack() {
        return null;
    }
}
