package com.liquidnet.service.sweet.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.liquidnet.commons.lang.util.IDGenerator;
import com.liquidnet.service.sweet.entity.SweetLostFoundItem;
import com.liquidnet.service.sweet.mapper.SweetLostFoundItemMapper;
import com.liquidnet.service.sweet.param.SweetLostFoundItemParam;
import com.liquidnet.service.sweet.service.ISweetLostFoundItemService;
import com.liquidnet.service.sweet.utils.LostFoundRedisUtils;
import com.liquidnet.service.sweet.vo.SweetLostFoundItemVo;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 失物招领信息服务实现类
 *
 * @author liquidnet
 * @since 2025-01-18
 */
@Service
public class SweetLostFoundItemServiceImpl extends ServiceImpl<SweetLostFoundItemMapper, SweetLostFoundItem> implements ISweetLostFoundItemService {

    @Autowired
    private LostFoundRedisUtils lostFoundRedisUtils;

    @Override
    public boolean publishItem(SweetLostFoundItemParam item) {
        if (item == null) {
            return false;
        }

        SweetLostFoundItem lostFoundItem = SweetLostFoundItem.getNew();
        lostFoundItem.setItemId(IDGenerator.nextSnowId());
        BeanUtils.copyProperties(item, lostFoundItem);

        lostFoundRedisUtils.deleteItemListCache(item.getPerformanceId());

        return baseMapper.insert(lostFoundItem) > 0;
    }

    @Override
    public boolean editItem(SweetLostFoundItemParam item) {
        if (item == null || item.getId() == null) {
            return false;
        }

        SweetLostFoundItem lostFoundItem = SweetLostFoundItem.getNew();
        BeanUtils.copyProperties(item, lostFoundItem);

        lostFoundRedisUtils.deleteItemListCache(item.getPerformanceId());
        lostFoundRedisUtils.deleteItemDetailCache(item.getId());

        return baseMapper.updateById(lostFoundItem) > 0;
    }

    @Override
    public boolean deleteItem(Long id) {
        if (id == null) {
            return false;
        }

        // 检查物品是否存在
        SweetLostFoundItem item = baseMapper.selectById(id);
        if (item == null || item.getIsDeleted() == 1) {
            return false;
        }

        item.setIsDeleted(1);

        lostFoundRedisUtils.deleteItemListCache(item.getPerformanceId());

        return baseMapper.updateById(item) > 0;
    }

    @Override
    public SweetLostFoundItemVo getItemDetail(Long id) {
        if (id == null) {
            return null;
        }

        SweetLostFoundItemVo itemDetail = lostFoundRedisUtils.getItemDetail(id);
        if (itemDetail == null) {
            SweetLostFoundItem item = baseMapper.selectById(id);
            if (item == null || item.getIsDeleted() == 1) {
                return null;
            }

            SweetLostFoundItemVo vo = new SweetLostFoundItemVo();
            BeanUtils.copyProperties(item, vo);
            lostFoundRedisUtils.setItemDetail(id, vo);
            return vo;
        }

        return itemDetail;
    }

    @Override
    public List<SweetLostFoundItemVo> getItemList(Integer itemType, String performanceId) {
        List<SweetLostFoundItemVo> itemList = lostFoundRedisUtils.getItemList(performanceId, itemType);
        if (itemList == null || itemList.isEmpty()) {
            QueryWrapper<SweetLostFoundItem> queryWrapper = new QueryWrapper<>();

            // 过滤
            queryWrapper.eq("is_deleted", 0);
            queryWrapper.eq("performance_id", performanceId.trim());

            // 排序
            queryWrapper.orderByDesc("create_time");

            // 执行查询
            List<SweetLostFoundItem> items = baseMapper.selectList(queryWrapper);

            // 转换为VO对象列表
            List<SweetLostFoundItemVo> itemVos = items.stream()
                    .map(item -> {
                        SweetLostFoundItemVo vo = new SweetLostFoundItemVo();
                        BeanUtils.copyProperties(item, vo);
                        return vo;
                    })
                    .collect(Collectors.toList());

            lostFoundRedisUtils.setItemList(performanceId, itemVos);

            if(null == itemType) {
                return itemVos;
            }
            return itemVos.stream()
                    .filter(item -> itemType.equals(item.getItemType()))
                    .collect(Collectors.toList());
        }

        return itemList;
    }
}