package com.liquidnet.service.kylin.utils;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.liquidnet.common.cache.redis.util.RedisUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.kylin.constant.KylinRedisConst;
import com.liquidnet.service.kylin.dto.vo.*;
import com.liquidnet.service.kylin.dto.vo.mongo.*;
import com.liquidnet.service.kylin.dto.vo.returns.KylinOrderListVo;
import com.liquidnet.service.kylin.entity.*;
import com.liquidnet.service.kylin.mapper.KylinBuyNoticeMapper;
import io.lettuce.core.GeoArgs;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Component;

import java.util.List;

@Component
public class DataUtils {

    @Autowired
    private KylinBuyNoticeMapper buyNoticeMapper;
    @Autowired
    private MongoTemplate mongoTemplate;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private MongoVoUtils mongoVoUtils;

    /**
     * 根据 购票须知 id 获取 购票须知文案
     *
     * @param buyNoticeIds 购票须知 数组
     * @return 购票须知 json串
     */
    public String getBuyNoticeJsonString(List<String> buyNoticeIds) {
        List<KylinBuyNotice> data = buyNoticeMapper.selectList(new UpdateWrapper<KylinBuyNotice>().in("buy_notice_id", buyNoticeIds));
        return JsonUtils.toJson(data);
    }

    /**
     * 初始化普通库存
     *
     * @param ticketId     票id
     * @param totalGeneral 普通库存
     */
    public void setSurplusGeneral(String ticketId, int totalGeneral) {
        redisUtil.set(KylinRedisConst.PERFORMANCES_INVENTORY + ticketId + ":" + KylinRedisConst.SURPLUS_GENERAL, totalGeneral);
    }

    /**
     * 获取普通剩余库存
     *
     * @param ticketId 票id
     * @return 普通剩余库存
     */
    public int getSurplusGeneral(String ticketId) {
        return (int) redisUtil.get(KylinRedisConst.PERFORMANCES_INVENTORY + ticketId + ":" + KylinRedisConst.SURPLUS_GENERAL);
    }

    /**
     * 修改普通库存 并 返回修改后的数量
     *
     * @param ticketId       票id
     * @param surplusGeneral 普通库存
     * @return 普通剩余库存
     */
    public int changeSurplusGeneral(String ticketId, int surplusGeneral) {
        if (surplusGeneral > 0) {
            return (int) redisUtil.incr(KylinRedisConst.PERFORMANCES_INVENTORY + ticketId + ":" + KylinRedisConst.SURPLUS_GENERAL, surplusGeneral);
        } else {
            return (int) redisUtil.decr(KylinRedisConst.PERFORMANCES_INVENTORY + ticketId + ":" + KylinRedisConst.SURPLUS_GENERAL, Math.abs(surplusGeneral));
        }
    }

    /**
     * 初始化兑换库存
     *
     * @param ticketId      票id
     * @param totalExchange 兑换库存
     */
    public void setSurplusExchange(String ticketId, int totalExchange) {
        redisUtil.set(KylinRedisConst.PERFORMANCES_INVENTORY + ticketId + ":" + KylinRedisConst.SURPLUS_EXCHANGE, totalExchange);
    }

    /**
     * 获取兑换剩余库存
     *
     * @param ticketId 票id
     * @return 兑换剩余库存
     */
    public int getSurplusExchange(String ticketId) {
        return (int) redisUtil.get(KylinRedisConst.PERFORMANCES_INVENTORY + ticketId + ":" + KylinRedisConst.SURPLUS_EXCHANGE);
    }

    /**
     * 变更兑换库存 并 返回 修改后的数量
     *
     * @param ticketId        票id
     * @param surplusExchange 普通库存
     * @return 兑换剩余库存
     */
    public int changeSurplusExchange(String ticketId, int surplusExchange) {
        if (surplusExchange > 0) {
            return (int) redisUtil.incr(KylinRedisConst.PERFORMANCES_INVENTORY + ticketId + ":" + KylinRedisConst.SURPLUS_EXCHANGE, surplusExchange);
        } else {
            return (int) redisUtil.decr(KylinRedisConst.PERFORMANCES_INVENTORY + ticketId + ":" + KylinRedisConst.SURPLUS_EXCHANGE, Math.abs(surplusExchange));
        }
    }

    /**
     * 根据演出id 获取 演出vo 详情
     *
     * @param performanceId
     * @return
     */
    public KylinPerformanceVo getPerformanceVo(String performanceId) {
        if (redisUtil.hasKey(KylinRedisConst.PERFORMANCES + performanceId)) {
        } else {
            KylinPerformanceVo performanceData = mongoTemplate.findOne(Query.query(Criteria.where("performancesId").is(performanceId)), KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());
            redisUtil.set(KylinRedisConst.PERFORMANCES + performanceId, performanceData);
        }
        return (KylinPerformanceVo) redisUtil.get(KylinRedisConst.PERFORMANCES + performanceId);
    }

    public void setOrderTicketVo(String userId,List<KylinOrderListVo> vo) {
        redisUtil.set(KylinRedisConst.ORDER_LIST + userId, vo);
    }

    public List<KylinOrderListVo> getOrderList(String userId) {
        if (redisUtil.hasKey(KylinRedisConst.ORDER_LIST + userId)) {
        } else {
            List<KylinOrderListVo> voList = mongoTemplate.find(Query.query(Criteria.where("userId").is(userId))
                    .with(Sort.by(Sort.Direction.DESC, "createdAt")).limit(40),
                    KylinOrderListVo.class, KylinOrderTicketVo.class.getSimpleName());
            redisUtil.set(KylinRedisConst.ORDER_LIST + userId, voList);
        }
        return (List<KylinOrderListVo>) redisUtil.get(KylinRedisConst.ORDER_LIST + userId);
    }

    /**
     * 根据订单id 获取 订单vo 详情
     *
     * @param orderId
     * @return
     */
    public KylinOrderTicketVo getOrderTicketVo(String orderId) {
        if (redisUtil.hasKey(KylinRedisConst.ORDER + orderId)) {
        } else {
            KylinOrderTicketVo ticketData = mongoTemplate.findOne(Query.query(Criteria.where("orderTicketsId").is(orderId)), KylinOrderTicketVo.class, KylinOrderTicketVo.class.getSimpleName());
            List<KylinOrderTicketEntitiesVo> kylinOrderTicketEntitiesVoList = mongoTemplate.find(Query.query(Criteria.where("orderId").is(orderId)), KylinOrderTicketEntitiesVo.class, KylinOrderTicketEntitiesVo.class.getSimpleName());
            ticketData.setEntitiesVoList(kylinOrderTicketEntitiesVoList);
            redisUtil.set(KylinRedisConst.ORDER + orderId, ticketData);
        }
        return (KylinOrderTicketVo) redisUtil.get(KylinRedisConst.ORDER + orderId);
    }

    /**
     * 删除订单redis
     *
     * @param orderId
     */
    public void delOrderTicketRedis(String orderId) {
        redisUtil.del(KylinRedisConst.ORDER + orderId);
    }

    /**
     * 获取bannerVo
     * @param bannerId
     * @return
     */
    public KylinBannersVo getBannersVo(String bannerId) {
        if (redisUtil.hasKey(KylinRedisConst.BANNERS + bannerId)) {
        } else {
            KylinBannersVo ticketData = mongoTemplate.findOne(Query.query(Criteria.where("bannersId").is(bannerId)), KylinBannersVo.class, KylinBannersVo.class.getSimpleName());
            redisUtil.set(KylinRedisConst.BANNERS + bannerId, ticketData);
        }
        return (KylinBannersVo) redisUtil.get(KylinRedisConst.BANNERS + bannerId);
    }

    /**
     * 删除 banner redis
     *
     * @param bannerId
     */
    public void delBannersRedis(String bannerId) {
        redisUtil.del(KylinRedisConst.BANNERS + bannerId);
    }

    /**
     * 获取 推荐演出vo
     *
     * @param recommendId
     * @return
     */
    public KylinPerformanceVo getPerformanceRecommendVo(String recommendId) {
        if (redisUtil.hasKey(KylinRedisConst.PERFORMANCES_RECOMMEND + recommendId)) {
        } else {
            KylinPerformanceVo ticketData = mongoTemplate.findOne(Query.query(Criteria.where("performancesId").is(recommendId)), KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());
            redisUtil.set(KylinRedisConst.PERFORMANCES_RECOMMEND + recommendId, ticketData);
        }
        return (KylinPerformanceVo) redisUtil.get(KylinRedisConst.PERFORMANCES_RECOMMEND + recommendId);
    }

    /**
     * 删除 推荐演出 redis
     *
     * @param recommendId
     */
    public void delPerformanceRecommendRedis(String recommendId) {
        redisUtil.del(KylinRedisConst.PERFORMANCES_RECOMMEND + recommendId);
    }

    /**
     * 获取 验票账号 vo
     *
     * @param checkUserId
     * @return
     */
    public KylinCheckUserVo getCheckUserVo(String checkUserId) {
        if (redisUtil.hasKey(KylinRedisConst.CHECK_USER + checkUserId)) {
        } else {
            KylinCheckUserVo ticketData = mongoTemplate.findOne(Query.query(Criteria.where("checkUserId").is(checkUserId)), KylinCheckUserVo.class, KylinCheckUserVo.class.getSimpleName());
            redisUtil.set(KylinRedisConst.CHECK_USER + checkUserId, ticketData);
        }
        return (KylinCheckUserVo) redisUtil.get(KylinRedisConst.CHECK_USER + checkUserId);
    }

    /**
     * 删除 验票账号 redis
     *
     * @param checkUserId
     */
    public void delCheckUserRedis(String checkUserId) {
        redisUtil.del(KylinRedisConst.CHECK_USER + checkUserId);
    }

    /**
     * 获取 验票账号关系 vo
     *
     * @param checkUserId
     * @return
     */
    public KylinCheckUserPerformanceVo getCheckUserRelationVo(String checkUserId) {
        if (redisUtil.hasKey(KylinRedisConst.CHECK_USER_RELATION + checkUserId)) {
        } else {
            KylinCheckUserPerformanceVo ticketData = mongoTemplate.findOne(Query.query(Criteria.where("checkUserId").is(checkUserId)), KylinCheckUserPerformanceVo.class, KylinCheckUserPerformanceVo.class.getSimpleName());
            redisUtil.set(KylinRedisConst.CHECK_USER_RELATION + checkUserId, ticketData);
        }
        return (KylinCheckUserPerformanceVo) redisUtil.get(KylinRedisConst.CHECK_USER_RELATION + checkUserId);
    }

    /**
     * 删除 验票账号关系 vo
     *
     * @param checkUserId
     */
    public void delCheckUserRelationRedis(String checkUserId) {
        redisUtil.del(KylinRedisConst.CHECK_USER_RELATION + checkUserId);
    }

    /**
     * 获取入场须知
     *
     * @param enterInfoId
     * @return
     */
    public String getEnterInfo(String enterInfoId) {
        if (redisUtil.hasKey(KylinRedisConst.ENTER_INFO + enterInfoId)) {
        } else {
            return "入场须知";
        }
        return (String) redisUtil.get(KylinRedisConst.ENTER_INFO + enterInfoId);
    }


    public void updatePerformanceMongo(String performanceIds) {
        //查询 mysql 数据
        KylinPerformanceVo vo = mongoVoUtils.combinePerformanceVoData(performanceIds);
        if (vo != null) {
            try {
                //删除 mongo数据
                mongoTemplate.remove(Query.query(Criteria.where("performancesId").is(performanceIds)), KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());
            } catch (Exception e) {
                e.printStackTrace();
            }
            try {
                //新增 mongo数据
                mongoTemplate.insert(vo, KylinPerformanceVo.class.getSimpleName());
            } catch (Exception e) {
                e.printStackTrace();
            }
            try {
                //删除redis
                redisUtil.del(KylinRedisConst.PERFORMANCES + performanceIds);
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else {
            System.out.println("创建Vo异常");
        }
    }
}
