package com.liquidnet.client.admin.web.controller.zhengzai.sweet;

import com.github.pagehelper.PageInfo;
import com.liquidnet.client.admin.common.annotation.Log;
import com.liquidnet.client.admin.common.core.controller.BaseController;
import com.liquidnet.client.admin.common.core.domain.AjaxResult;
import com.liquidnet.client.admin.common.core.page.TableDataInfo;
import com.liquidnet.client.admin.common.enums.BusinessType;
import com.liquidnet.client.admin.zhengzai.sweet.service.ISweetIntegralActivityService;
import com.liquidnet.service.sweet.dto.SweetIntegralActivityDto;
import com.liquidnet.service.sweet.dto.param.admin.SweetIntegralActivityFromParam;
import com.liquidnet.service.sweet.dto.param.admin.SweetIntegralActivityListSearchParam;
import com.liquidnet.service.sweet.dto.vo.admin.SweetIntegralActivityVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.Max;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;

/**
 * <p>
 * 积分活动表 前端控制器
 * </p>
 *
 * @author jiangxiulong
 * @since 2021-10-20
 */
@Api(tags = "积分活动管理")
@Validated
@Controller
@RequestMapping("/sweet/integralActivity")
public class SweetIntegralActivityController extends BaseController {
    private String prefix = "zhengzai/sweet/integralActivity";

    @Value("${liquidnet.client.admin.platformUrl}")
    private String platformUrl;

    @Autowired
    private ISweetIntegralActivityService iSweetIntegralActivityService;

    @Log(title = "积分活动创建", businessType = BusinessType.INSERT)
    @RequiresPermissions("sweet:integralActivity:create")
    @PostMapping("create")
    @ApiOperation(value = "积分活动创建")
    @ResponseBody
    public AjaxResult create(@Valid @RequestBody SweetIntegralActivityFromParam param) {
        Integer result = iSweetIntegralActivityService.createForm(param);
        if (result > 0) {
            return success("操作成功");
        } else {
            return error("操作失败");
        }
    }

    @Log(title = "积分活动编辑", businessType = BusinessType.UPDATE)
    @RequiresPermissions("sweet:integralActivity:update")
    @PutMapping("update")
    @ApiOperation(value = "积分活动编辑")
    @ResponseBody
    public AjaxResult update(@Valid @RequestBody SweetIntegralActivityFromParam param) {
        Integer result = iSweetIntegralActivityService.updateForm(param);
        if (result > 0) {
            return success("操作成功");
        } else {
            return error("操作失败");
        }
    }

    @Log(title = "积分活动上下线", businessType = BusinessType.UPDATE)
    @RequiresPermissions("sweet:integralActivity:isOnline")
    @ApiOperation(value = "积分活动上下线")
    @PutMapping("isOnline")
    @ResponseBody
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "integralActivityId", value = "活动id"),
            @ApiImplicitParam(type = "form", required = true, dataType = "Integer", name = "isOnline", value = "是否上线 1下线 2上线"),
    })
    public AjaxResult isOnline(
            @RequestParam("integralActivityId") @NotEmpty(message = "integralActivityId不能为空") String integralActivityId,
            @RequestParam("isOnline") @NotNull(message = "isOnline不能为空") @Min(value = 1, message = "isOnline无效") @Max(value = 2, message = "isOnline无效") Integer isOnline
    ) {
        Integer result = iSweetIntegralActivityService.isOnline(integralActivityId, isOnline);
        if (result > 0) {
            return success("操作成功");
        } else {
            return error("操作失败");
        }
    }

    @Log(title = "积分活动详情", businessType = BusinessType.DETAIL)
    @GetMapping("details")
    @ApiOperation(value = "积分活动详情")
    @ResponseBody
    @ApiImplicitParams({
            @ApiImplicitParam(type = "query", required = true, dataType = "String", name = "integralActivityId", value = "活动id"),
    })
    public AjaxResult details(@RequestParam("integralActivityId") String integralActivityId) {
        SweetIntegralActivityVo result = iSweetIntegralActivityService.detail(integralActivityId);
        return AjaxResult.success(result);
    }

    @Log(title = "积分活动列表数据", businessType = BusinessType.LIST)
    @RequiresPermissions("sweet:integralActivity:list")
    @ApiOperation(value = "积分活动列表数据")
    @PostMapping("list")
    @ResponseBody
    public TableDataInfo list(SweetIntegralActivityListSearchParam listParam) {
        PageInfo<SweetIntegralActivityDto> result = iSweetIntegralActivityService.list(listParam);
        return getDataTable(result.getList());
    }

    @GetMapping("listView")
    @ApiOperation(value = "积分活动列表view")
    public String listView() {
        return prefix + "/list";
    }

    @GetMapping("formView")
    @ApiOperation(value = "积分活动添加编辑view")
    public String formView(ModelMap mmap) {
        mmap.put("platformUrl", platformUrl);
        return prefix + "/form";
    }

}
