package com.liquidnet.service.candy.controller;


import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.candy.service.ICandyCouponService;
import com.liquidnet.service.candy.vo.CandyCouponVo;
import com.liquidnet.service.candy.vo.CandyMyCouponListVo;
import com.liquidnet.service.candy.vo.CandyUseResultVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;

/**
 * <p>
 * 券基础信息 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2021-08-18
 */
@Api(tags = "券相关接口")
@RestController
@Validated
@RequestMapping("/candy-coupon")
public class CandyCouponController {

    @Autowired
    private ICandyCouponService candyCouponService;

    //我的券列表
    @GetMapping("my")
    @ApiOperation("我的券列表")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "Integer", name = "type", value = "类型 1可用 2过期/已使用 ", required = true),
    })
    public ResponseDto<CandyMyCouponListVo> myCoupon(@RequestParam("type") @NotNull Integer type) {
        return ResponseDto.success(candyCouponService.myCoupon(type));
    }

    //要使用券列表 - 演出
    @GetMapping("preUse/performance")
    @ApiOperation("要使用券列表 - 演出")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "BigDecimal", name = "priceTotal", value = "待支付总金额", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "performanceId", value = "演出id", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "timeId", value = "场次id", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "ticketId", value = "票id", required = true),
            @ApiImplicitParam(type = "form", dataType = "Integer", name = "type", value = "类型 1可用 2过期/已使用 ", required = true),
    })
    public ResponseDto<CandyMyCouponListVo> preUsePerformanceCoupon(@RequestParam("priceTotal") @NotNull BigDecimal priceTotal,
                                                                    @RequestParam("performanceId") @NotNull @NotBlank String performanceId,
                                                                    @RequestParam("timeId") @NotNull @NotBlank String timeId,
                                                                    @RequestParam("ticketId") @NotNull @NotBlank String ticketId,
                                                                    @RequestParam("type") @NotNull Integer type) {
        return ResponseDto.success(candyCouponService.preUsePerformanceCoupon(priceTotal, performanceId, timeId, ticketId, type));
    }

    // 是否可用券演出
    @GetMapping("preCanUse/performance")
    @ApiOperation("是否可用券 - 演出")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "BigDecimal", name = "priceTotal", value = "待支付总金额", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "performanceId", value = "演出id", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "timeId", value = "场次id", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "ticketId", value = "票id", required = true),
    })
    public ResponseDto<HashMap<String, Integer>> preCanUsePerformanceCoupon(@RequestParam("priceTotal") @NotNull BigDecimal priceTotal,
                                                                            @RequestParam("performanceId") @NotNull @NotBlank String performanceId,
                                                                            @RequestParam("timeId") @NotNull @NotBlank String timeId,
                                                                            @RequestParam("ticketId") @NotNull @NotBlank String ticketId) {
        HashMap<String, Integer> hashMap = CollectionUtil.mapStringInteger();
        hashMap.put("canUse", candyCouponService.preCanUsePerformanceCoupon(priceTotal, performanceId, timeId, ticketId));
        return ResponseDto.success(hashMap);
    }


    //要使用券列表 - 商品
    @GetMapping("preUse/goods")
    @ApiOperation("要使用券列表 - 商品")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "BigDecimal", name = "priceTotal", value = "待支付总金额", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "goodId", value = "演出id", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "produceId", value = "场次id", required = true),
            @ApiImplicitParam(type = "form", dataType = "Integer", name = "type", value = "类型 1可用 2过期/已使用 ", required = true),
    })
    public ResponseDto<CandyMyCouponListVo> preUseGoodCoupon(@RequestParam("priceTotal") @NotNull BigDecimal priceTotal,
                                                             @RequestParam("goodId") @NotNull @NotBlank String goodId,
                                                             @RequestParam("produceId") @NotNull @NotBlank String produceId,
                                                             @RequestParam("type") @NotNull Integer type) {
        return ResponseDto.success(candyCouponService.preUseGoodCoupon(priceTotal, goodId, produceId, type));
    }

    //是否可用券 - 商品
    @GetMapping("preCanUse/goods")
    @ApiOperation("是否可用券 - 商品")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "BigDecimal", name = "priceTotal", value = "待支付总金额", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "goodId", value = "演出id", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "produceId", value = "场次id", required = true),
    })
    public ResponseDto<HashMap<String, Integer>> preCanUseGoodCoupon(@RequestParam("priceTotal") @NotNull BigDecimal priceTotal,
                                                                     @RequestParam("goodId") @NotNull @NotBlank String goodId,
                                                                     @RequestParam("produceId") @NotNull @NotBlank String produceId) {
        HashMap<String, Integer> hashMap = CollectionUtil.mapStringInteger();
        hashMap.put("canUse", candyCouponService.preCanUseGoodCoupon(priceTotal, goodId, produceId));
        return ResponseDto.success(hashMap);
    }


    //要使用券列表 - 演出提前
    @GetMapping("my/advance")
    @ApiOperation("提前券列表")
    public ResponseDto<List<CandyCouponVo>> myAdvanceCoupon() {
        return ResponseDto.success(candyCouponService.myAdvanceCoupon());
    }

    //判断券是否可用
    @GetMapping("state")
    @ApiOperation("券是否可用")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "uCouponId", value = "券唯一表示id", required = true),
    })
    public ResponseDto<HashMap<String, Integer>> stateCoupon(@RequestParam("uCouponId") @NotNull @NotBlank String uCouponId) {
        Integer result = candyCouponService.stateCoupon(uCouponId);
        if (result == null) {
            return ResponseDto.failure();
        }
        HashMap<String, Integer> hashMap = CollectionUtil.mapStringInteger();
        hashMap.put("state", result);
        return ResponseDto.success(hashMap);
    }

    //判断券是否可用并使用
    @PostMapping("use")
    @ApiOperation("券使用")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "uCouponId", value = "券唯一表示id", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "content", value = "消费内容", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "totalPrice", value = "消费价格", required = true),
    })
    public ResponseDto<CandyUseResultVo> useCoupon(@RequestParam("uCouponId") @NotNull @NotBlank String uCouponId,
                                                   @RequestParam("content") @NotNull @NotBlank String content,
                                                   @RequestParam("totalPrice") @NotNull String totalPrice) {
        CandyUseResultVo result = candyCouponService.useCoupon(uCouponId, content, totalPrice);
        if (result == null) {
            return ResponseDto.failure();
        }
        return ResponseDto.success(result);
    }

    //回退券
    @PostMapping("useBack")
    @ApiOperation("回退券")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "uCouponId", value = "券唯一表示id(逗号隔开的字符串)", required = true),
    })
    public ResponseDto<Boolean> useBackCoupon(@RequestParam("uCouponId") @NotNull @NotBlank String uCouponId) {
        candyCouponService.useBackCoupon(uCouponId);
        return ResponseDto.success();
    }

    //兑换券
    @PostMapping("receive")
    @ApiOperation("领取券")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "ccode", value = "兑换码", required = true),
    })
    public ResponseDto<Boolean> receiveCoupon(@RequestParam("ccode") @NotNull @NotBlank String ccode) {
        return candyCouponService.receiveCoupon(ccode);
//        if (result) {
//            return ResponseDto.success();
//        } else {
//            return ResponseDto.failure();
//        }
    }

    //兑换券详情
    @PostMapping("receive/details")
    @ApiOperation("领取券详情")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "ccode", value = "兑换码", required = true),
    })
    public ResponseDto<CandyCouponVo> receiveCouponDetails(@RequestParam("ccode") @NotNull @NotBlank String ccode) {
        return candyCouponService.receiveCouponDetails(ccode);
//        if (result != null) {
//            return ResponseDto.success(result);
//        } else {
//            return ResponseDto.failure();
//        }
    }
}
