package com.liquidnet.service.adam.service.impl;

import com.fasterxml.jackson.databind.JsonNode;
import com.liquidnet.common.exception.LiquidnetServiceException;
import com.liquidnet.commons.lang.util.*;
import com.liquidnet.service.adam.dto.AdamEntersParam;
import com.liquidnet.service.adam.dto.vo.AdamEntersVo;
import com.liquidnet.service.adam.service.AdamRdmService;
import com.liquidnet.service.adam.service.IAdamEntersService;
import com.liquidnet.service.adam.util.QueueUtils;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.SqlMapping;
import com.liquidnet.service.base.constant.MQConst;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.IntStream;

/**
 * <p>
 * 入场人 服务实现类
 * </p>
 *
 * @author liquidnet
 * @since 2021-04-28
 */
@Slf4j
@Service
public class AdamEntersServiceImpl implements IAdamEntersService {
    @Autowired
    Environment env;
    @Autowired
    QueueUtils queueUtils;
    @Autowired
    AdamRdmService adamRdmService;

    @Override
//    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public String add(AdamEntersParam parameter) {
        String currentUid = CurrentUtil.getCurrentUid();
        LocalDateTime now = LocalDateTime.now();


        if (1 == parameter.getType()) {
            if (!adamRdmService.isCertification(1, parameter.getIdCard(), parameter.getName())) {
                identityHandler(currentUid, parameter.getName(), parameter.getIdCard());

                adamRdmService.setCertification(1, parameter.getIdCard(), parameter.getName());
            }
        }


        List<AdamEntersVo> vos = adamRdmService.getEntersVoByUid(currentUid);

        AdamEntersVo vo = AdamEntersVo.getNew();
        BeanUtils.copyProperties(parameter, vo);
        vo.setEntersId(IDGenerator.nextSnowId());
        vo.setUid(currentUid);
        vo.setIsDefault(CollectionUtils.isEmpty(vos));
        vo.setState(1);
        vo.setCreatedAt(now);

        long s = System.currentTimeMillis();
//        adamRdmService.delEntersVoByUid(currentUid);
        adamRdmService.addEntersVoByUid(currentUid, vos, vo);
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        String msg = SqlMapping.get("adam_enters.add",
                vo.getEntersId(), vo.getUid(), vo.getType(), vo.getName(), vo.getMobile(), vo.getIdCard(), vo.getIsDefault(), vo.getState(), now
        );
        log.debug("#SQL.GET耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        queueUtils.sendMsgByRedis(
                MQConst.AdamQueue.SQL_UCENTER.getKey(),
                msg
        );
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);
        return vo.getEntersId();
    }

    @Override
//    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void def(String uid, String entersId) {
        LocalDateTime now = LocalDateTime.now();

        LinkedList<Object[]> toMqObjs = CollectionUtil.linkedListObjectArr();
        List<AdamEntersVo> vos = adamRdmService.getEntersVoByUid(uid);
        if (vos.size() > 1) {// 取消原默认
            AdamEntersVo defaultVo = queryDefault(vos, uid, false);
            defaultVo.setIsDefault(false);
            defaultVo.setUpdatedAt(now);

            toMqObjs.add(new Object[]{false, now, defaultVo.getEntersId()});

            vos = this.collectionProcess(vos, defaultVo.getEntersId(), defaultVo);
        }
        {// 设置新默认
            AdamEntersVo defaultVoAfter = adamRdmService.getEntersVoByUidEntersId(vos, entersId);
            defaultVoAfter.setIsDefault(true);
            defaultVoAfter.setUpdatedAt(now);

            toMqObjs.add(new Object[]{true, now, entersId});

            vos = this.collectionProcess(vos, defaultVoAfter.getEntersId(), defaultVoAfter);
        }
        if (!CollectionUtils.isEmpty(toMqObjs)) {
            long s = System.currentTimeMillis();
            adamRdmService.setEntersVoByUid(uid, vos);
            log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);
            s = System.currentTimeMillis();
            queueUtils.sendMsgByRedis(
                    MQConst.AdamQueue.SQL_UCENTER.getKey(),
                    SqlMapping.get("adam_enters.update.is_default", toMqObjs)
            );
            log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);
        }
    }

    @Override
//    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void edit(AdamEntersParam parameter) {
        String currentUid = CurrentUtil.getCurrentUid();
        LocalDateTime now = LocalDateTime.now();

        if (1 == parameter.getType()) {
            if (!adamRdmService.isCertification(1, parameter.getIdCard(), parameter.getName())) {
                identityHandler(currentUid, parameter.getName(), parameter.getIdCard());
            }
        }

        List<AdamEntersVo> vos = adamRdmService.getEntersVoByUid(currentUid);
        AdamEntersVo updateVo = adamRdmService.getEntersVoByUidEntersId(vos, parameter.getEntersId());
        updateVo.setType(parameter.getType());
        updateVo.setName(parameter.getName());
        updateVo.setMobile(parameter.getMobile());
        updateVo.setIdCard(parameter.getIdCard());
        updateVo.setUpdatedAt(now);
        long s = System.currentTimeMillis();
        adamRdmService.setEntersVoByUid(currentUid, this.collectionProcess(vos, parameter.getEntersId(), updateVo));
        if (1 == parameter.getType()) {
            adamRdmService.setCertification(1, parameter.getIdCard(), parameter.getName());
        }
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        queueUtils.sendMsgByRedis(
                MQConst.AdamQueue.SQL_UCENTER.getKey(),
                SqlMapping.get("adam_enters.edit",
                        updateVo.getType(), updateVo.getName(), updateVo.getMobile(), updateVo.getIdCard(), updateVo.getIsDefault(), updateVo.getState(), now, updateVo.getEntersId()
                )
        );
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);
    }

    @Override
//    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void remove(String uid, String entersId) {
        LocalDateTime now = LocalDateTime.now();
        adamRdmService.rmvEntersVoByUid(uid, adamRdmService.getEntersVoByUid(uid), entersId);

        long s = System.currentTimeMillis();
        queueUtils.sendMsgByRedis(
                MQConst.AdamQueue.SQL_UCENTER.getKey(),
                SqlMapping.get("adam_enters.remove", now, now, entersId)
        );
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);
    }

    @Override
    public AdamEntersVo queryDefault(List<AdamEntersVo> vos, String uid, boolean downgrade) {
        if (CollectionUtils.isEmpty(vos)) {
            vos = adamRdmService.getEntersVoByUid(uid);
        }

        if (!CollectionUtils.isEmpty(vos)) {
            AdamEntersVo defaultVo = null;
            for (AdamEntersVo vo : vos) {
                if (vo.getIsDefault()) {
                    return vo;
                }
                if (downgrade) {
                    defaultVo = null == defaultVo ? vo :
                            vo.getCreatedAt().compareTo(defaultVo.getCreatedAt()) > 0 ? vo : defaultVo;
                }
            }
            return defaultVo;
        }
        return null;
    }

    /* ---------------------------------------------------------------  */

    private List<AdamEntersVo> collectionProcess(List<AdamEntersVo> vos, String replaceId, AdamEntersVo updateVo) {
        long s = System.currentTimeMillis();
        int idx = IntStream.range(0, vos.size())
                .filter(i -> vos.get(i).getEntersId().equals(replaceId))
                .findFirst().orElse(-1);
        if (idx == -1) {
            ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("10017");
            throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
        }
        vos.set(idx, updateVo);
        log.debug("#collect.process耗时:{}ms", System.currentTimeMillis() - s);
        return vos;
    }

    private void identityHandler(String currentUid, String name, String idCard) {
        String respStr = IdentityUtils.aliThird(name, idCard);
        JsonNode respJNode = JsonUtils.fromJson(respStr, JsonNode.class);
        if (null == respJNode || !"0".equals(respJNode.get("error_code").asText())) {
            log.info("###实名认证失败[{}]", respStr);
            ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("10102");
            throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
        }
    }
}
