package com.liquidnet.common.swagger.controller;

import com.liquidnet.common.swagger.annotation.ApiLevel;
import com.liquidnet.common.swagger.dto.ApiLevelInfoDto;
import com.liquidnet.common.swagger.dto.ApiLevelStatisticsDto;
import com.liquidnet.common.swagger.dto.MultiServiceStatisticsDto;
import com.liquidnet.common.swagger.service.MultiServiceApiLevelStatisticsService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 接口分级统计控制器
 * 
 * @author system
 * @since 2024-10-29
 */
@Api(tags = "系统管理-接口分级统计")
@RestController
@RequestMapping("/levelStatistics")
@Slf4j
public class ApiLevelStatisticsController {
    
    @Autowired
    private MultiServiceApiLevelStatisticsService statisticsService;
    

    @GetMapping("/all")
    @ApiOperation("获取所有服务接口分级统计概览")
    public MultiServiceStatisticsDto getAllServicesStatistics() {
        try {
            return statisticsService.getAllServicesStatistics();
        } catch (Exception e) {
            log.error("获取所有服务分级统计失败", e);
            throw new RuntimeException("获取所有服务分级统计失败: " + e.getMessage());
        }
    }
    
    @GetMapping("/health")
    @ApiOperation("获取所有服务健康状态")
    public Map<String, Object> getServicesHealth() {
        try {
            Map<String, Boolean> healthStatus = statisticsService.getServicesHealth();
            Map<String, Object> result = new HashMap<>();
            result.put("services", healthStatus);
            result.put("totalServices", healthStatus.size());
            result.put("healthyServices", healthStatus.values().stream().mapToInt(v -> v ? 1 : 0).sum());
            return result;
        } catch (Exception e) {
            log.error("获取服务健康状态失败", e);
            throw new RuntimeException("获取服务健康状态失败: " + e.getMessage());
        }
    }
    
    @GetMapping("/levels")
    @ApiOperation("获取所有接口级别定义")
    public List<ApiLevelInfoDto> getApiLevels() {
        List<ApiLevelInfoDto> levels = new ArrayList<>();
        
        for (ApiLevel.Level level : ApiLevel.Level.values()) {
            ApiLevelInfoDto levelInfo = new ApiLevelInfoDto();
            levelInfo.setCode(level.name());                    // L1, L2, L3, L4, L88, L99
            levelInfo.setValue(level.getValue());               // 1, 2, 3, 4, 88, 99
            levelInfo.setLabel(level.getDesc());                // 核心, 重要, 普通, 低级, 无需分级, 未配置
            levelInfo.setDescription(level.getDetail());        // 详细描述
            levelInfo.setIsUnconfigured(level.isUnconfigured()); // 是否为未配置
            levelInfo.setIsNoGrading(level.isNoGrading());       // 是否为无需分级
            
            // 判断是否为业务级别 (L1-L4)
            boolean isBusinessLevel = level.getValue() >= 1 && level.getValue() <= 4;
            levelInfo.setIsBusinessLevel(isBusinessLevel);
            
            levels.add(levelInfo);
        }
        
        return levels;
    }
    

    @GetMapping("/service")
    @ApiOperation("获取本服务接口分级统计信息")
    public ApiLevelStatisticsDto getServiceStatistics() {
        try {
            return statisticsService.getLocalServiceStatistics();
        } catch (Exception e) {
            log.error("获取服务统计失败", e);
            throw new RuntimeException("获取服务统计失败: " + e.getMessage());
        }
    }
    

}